#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pthread.h>

#include "db.h"
#include "libfma.h"
#include "lf_fabric.h"
#include "lf_switch.h"
#include "lf_dflt_error.h"


struct qargs {
  char *name;
  lf_enclosure_t **epp;
  int rc;
};

static void *run_query(void *);
static void parse_all_enclosures(struct lf_fabric *fp);

int Delay = 30;
int Thresh = 0;
int Show_disabled = 0;
int Show_RX_timeout = 0;
int Show_good = 0;
int Show_abs = 0;
int Show_worst = 0;
int Show_frac = 0;
int Run_once = 0;
int Show_updown = 0;

void
parse_args(
  int argc,
  char **argv)
{
  int c;
  extern char *optarg;
  char *fms_run;
  char *db_name;

  fms_run = NULL;
  db_name = NULL;

  while ((c = getopt(argc, argv, "a1dgprt:i:fwH:R:N:V")) != EOF) switch (c) {

  case 'H':
    fprintf(stderr,
	"Please note: -H has been replaced with -R and is deprecated\n");
    /* FALLSTHROUGH */

  case 'R':
    fms_run = optarg;
    break;

  case 'N':
    db_name = optarg;
    break;

  case '1':
    Run_once = 1;
    break;

  case 't':	/* reporting threshold */
    Thresh = atoi(optarg);
    break;

  case 'i':	/* inter-read delay */
    Delay = atoi(optarg);
    break;

  case 'a':
    Show_abs = 1;
    break;

  case 'r':
    Show_RX_timeout = 1;
    break;

  case 'w':
    Show_worst = 1;
    break;

  case 'd':
    Show_disabled = 1;
    break;

  case 'g':
    Show_good = 1;
    break;

  case 'p':
    Show_updown = 1;
    break;

  case 'f':
    Show_frac = 1;
    Run_once = 1;
    break;

  case 'V':
    printf("FMS version is %s\n", Lf_version);
    exit(0);
    break;
  }

  /* set dirs - doesn't care if they are NULL */
  lf_simple_set_fabric_dirs(fms_run, db_name);
}

#define CHECK_COUNTS(F,T) 					\
  do {								\
    int _p, _wp=0;						\
    int _sw, _wsw=0;						\
    int _slot, _wslot=0;					\
    int _x=0, _wx;						\
    int _d;							\
    int _worst;							\
    struct lf_enclosure *_ep;					\
    struct lf_linecard *_sp;					\
    struct lf_xbarport_data *_xdp;				\
    struct lf_xbar *_xp;					\
    union lf_node *_np;						\
    _worst = 0;							\
    for (_sw = 0; _sw < fp->num_enclosures; ++_sw) {		\
      _ep = fp->enclosures[_sw];				\
      for (_slot=0; _slot<_ep->num_slots; ++_slot) {		\
	_sp = _ep->slots[_slot];				\
	if (_sp == NULL) continue;				\
        for (_x=0; _x<_sp->num_xbars; ++_x) {			\
	  _xp = LF_XBAR(_sp->xbars[_x]);			\
	  for (_p = 0; _p < _xp->num_ports; ++_p) {		\
	    _xdp = _xp->data + _p;				\
	    _d = _xdp->cur_vals.F - _xdp->old_vals.F;		\
	    if (_d > Thresh) {					\
	      struct lf_xcvr *_xcp;				\
	      _np = _xp->phys_ports[_p];			\
	      printf("%s, slot %d, xbar %d port %d %s delta: %d", \
		_ep->name, lf_slot_display_no(_sp),	 	\
		_x, _p, T, _d); 				\
	      if (Show_abs) printf(" (=%d-%d)", \
		(int) _xdp->cur_vals.F, (int) _xdp->old_vals.F); \
	      if (_np == NULL) {				\
		printf(" (disconnected?)");			\
	      } else if (_np->ln_type != LF_NODE_LC_XCVR) {	\
		print_internal_link(_np, _xp->phys_rports[_p]);	\
	      } else if (LF_XCVR(_np)->p.linecard->slot != _slot) { \
		_xcp = LF_XCVR(_np);				\
		printf(" (slot %d, port %d)",			\
		  lf_slot_display_no(_xcp->p.linecard),		\
			_xcp->port);				\
	      }							\
	      _np = _xp->topo_ports[_p];                        \
	      if (_np != NULL && _np->ln_type == LF_NODE_NIC) {    \
		printf(" (%s)", LF_NIC(_np)->host->hostname);	\
              }							\
	      printf("\n");					\
	    }							\
	    if (_d > _worst) {					\
	      _worst = _d;					\
	      _wx = _x;						\
	      _wp = _p;						\
	      _wsw = _sw;					\
	      _wslot = _slot;					\
	    }							\
	  }							\
	}							\
      }								\
    }								\
    if (Show_worst && _worst > 0) {				\
      printf("worst %s on %s, slot %d, xbar %d, port %d : %d\n", \
	T, fp->enclosures[_wsw]->name, _wslot, _x, _wp, _worst); \
    }								\
  } while (0)

void
print_internal_link(
  union lf_node *np,
  int port)
{
  struct lf_xbar *xp;
  struct lf_linecard *lp;

  xp = LF_XBAR(np);
  lp = xp->linecard;
  printf(" (INTERNAL to slot %d,", lf_slot_display_no(lp));
  if (lp->num_xbars > 1) printf(" xbar %d,", xp->xbar_no);
  printf(" port %d)", port);
}

void
show_frac(
  struct lf_fabric *fp)
{
  int e;
  uint64_t g, b;

  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    int slot;

    ep = fp->enclosures[e];

    for (slot=0; slot<ep->num_slots; ++slot) {
      struct lf_linecard *lp;
      int x;

      lp = ep->slots[slot];
      if (lp == NULL) continue;

      /* check all xbars */
      for (x=0; x<lp->num_xbars; ++x) {
	struct lf_xbar *xp;
	int p;

	xp = LF_XBAR(lp->xbars[x]);
	for (p=0; p<xp->num_ports; ++p) {
	  double frac;

	  b = xp->data[p].cur_vals.badcrcs;
	  g = xp->data[p].cur_vals.goodcrcs;
	  if (g>0) {
	    frac = b * 100. / g;
	  } else {
	    frac = 0.;
	  }

	  if (frac < .05) continue;

	  printf("%s, slot %d, xbar %d port %d fraction is %.2f%%%s\n",
		ep->name, lf_slot_display_no(lp), x, p, frac,
		xp->data[p].cur_vals.control?"  (disabled)":"");
	}
      }
    }
  }
}

void
show_updown(
  struct lf_fabric *fp)
{
  int e;

  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    int slot;

    ep = fp->enclosures[e];

    for (slot=0; slot<ep->num_slots; ++slot) {
      struct lf_linecard *lp;
      int x;

      lp = ep->slots[slot];
      if (lp == NULL) continue;

      /* check all xbars */
      for (x=0; x<lp->num_xbars; ++x) {
	struct lf_xbar *xp;
	int p;

	xp = LF_XBAR(lp->xbars[x]);
	for (p=0; p<xp->num_ports; ++p) {

	  if (xp->data[p].cur_vals.portdown != xp->data[p].old_vals.portdown) {
	    if (xp->data[p].cur_vals.portdown) {
	      printf("%s, slot %d, xbar %d port %d is down\n",
		ep->name, lf_slot_display_no(lp), x, p);
	    } else {
	      printf("%s, slot %d, xbar %d port %d is back up\n",
		ep->name, lf_slot_display_no(lp), x, p);
	    }
	  }
	}
      }
    }
  }
}

void
show_disabled(
  struct lf_fabric *fp)
{
  int e;

  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    int slot;

    ep = fp->enclosures[e];

    for (slot=0; slot<ep->num_slots; ++slot) {
      struct lf_linecard *lp;
      int x;

      lp = ep->slots[slot];
      if (lp == NULL) continue;

      /* check all xbars */
      for (x=0; x<lp->num_xbars; ++x) {
	struct lf_xbar *xp;
	int p;

	xp = LF_XBAR(lp->xbars[x]);
	for (p=0; p<xp->num_ports; ++p) {

	  if (xp->data[p].cur_vals.control != 0) {
	    printf("%s, slot %d, xbar %d port %d disabled[%d]\n",
	      ep->name, lf_slot_display_no(lp), x, p,
	      xp->data[p].cur_vals.control);
	  }
	}
      }

      /* check all xcvrs */
      for (x=0; x<lp->num_xcvrs; ++x) {
	struct lf_xcvr *xcp;

	xcp = LF_XCVR(lp->xcvrs[x]);

	if (xcp->data->cur_vals.control != 0) {
	  printf("%s, slot %d, fiber port %d disabled[%d]\n",
	    ep->name, lf_slot_display_no(lp), x,
	    xcp->data->cur_vals.control);
	  }
      }
    }
  }
}

int
main(
  int argc,
  char **argv)
{
  struct lf_fabric *fp;
  struct lf_enclosure *ep;
  int i;

  lf_init();

  parse_args(argc, argv);

  fp = lf_simple_load_fabric();
  if (fp == NULL) LF_ERROR(("Error loading fabric"));

  printf("Watching:\n");
  for (i=0; i<fp->num_enclosures; ++i) {
    ep = fp->enclosures[i];
    printf("\t%s %s\n", ep->name, ep->product_id);
  }

  parse_all_enclosures(fp);
  parse_all_enclosures(fp);	/* re-read to make sure we're up-to-date */

  if (Show_disabled) {
    show_disabled(fp);
  }
  if (Show_frac) {
    show_frac(fp);
  }
  if (Run_once) return 0;

  while (1) {
    sleep(Delay);
    printf("==== %s ====\n", lf_timestamp());

    /* parse all enclosure files */
    parse_all_enclosures(fp);
    
    if (Show_good) {
      CHECK_COUNTS(goodcrcs, "Good CRC Count");
    }
    CHECK_COUNTS(badcrcs, "Bad CRC Count");
    if (Show_RX_timeout) {
      CHECK_COUNTS(receivetimeoutcount, "RX timeout Count");
    }
    CHECK_COUNTS(transmittimeoutcount, "TX timeout Count");
    /* CHECK_COUNTS(invalidroutes, "Invalid route Count"); */
    if (Show_disabled) {
      show_disabled(fp);
    }

    if (Show_updown) {
      show_updown(fp);
    }

    /*
     * Process the change list for each enclosure
     */
    for (i=0; i<fp->num_enclosures; ++i) {
      struct lf_switch_data_change *cp;
      struct lf_switch_data_change *ncp;
      int slot;

      ep = fp->enclosures[i];

      cp = ep->data->change_list;
      ep->data->change_list = NULL;
      while (cp != NULL) {
	ncp = cp->next;

	switch (cp->dc_type) {
	  struct lf_linecard *lp;

	  case LF_SWITCH_CHANGE_MISSING_LINECARD:
	    slot = cp->c.missing_linecard.slot;
	    printf("%s, slot %d (%s) is missing\n",
		ep->name, slot, ep->slots[slot]->product_id);
	    break;

	  case LF_SWITCH_CHANGE_NEW_LINECARD:
	    slot = cp->c.new_linecard.slot;
	    lp = lf_allocate_linecard(ep, slot, cp->c.new_linecard.product_id,
				      cp->c.new_linecard.serial_no);
	    if (lp == NULL) {
	      LF_ERROR(("Error allocating new linecard for %s, slot %d",
		         ep->name, slot));
	    }
	    printf("%s, new card at slot %d (%s)\n", 
		ep->name, lf_slot_display_no(lp),
		cp->c.new_linecard.product_id);
	    break;

	  case LF_SWITCH_CHANGE_CHANGED_LINECARD:
	    slot = cp->c.new_linecard.slot;
	    printf("%s, slot %d (%s) changed\n",
		ep->name, slot, cp->c.new_linecard.product_id);
	    break;
	}
	lf_free_switch_change(cp);
	cp = ncp;
      }
    }

  }

  return 0;

 except:
  return 1;
}


void
parse_all_enclosures(
  struct lf_fabric *fp)
{
  int i;
  pthread_t *tid;
  struct qargs *qa;
  int rc;

  /* if only 1, no thread - makes debugging easier, too. */
  if (fp->num_enclosures == 1) {
    lf_enclosure_t *ep;
    ep = fp->enclosures[0];
    rc = lf_query_switch(ep->name, &(fp->enclosures[0]));
    if (rc == -1) {
      LF_ERROR(("Error querying %s", ep->name));
    }
    return;
  }

  LF_CALLOC(tid, pthread_t, fp->num_enclosures);
  LF_CALLOC(qa, struct qargs, fp->num_enclosures);

  for (i=0; i<fp->num_enclosures; ++i) {
    qa[i].name = fp->enclosures[i]->name;
    qa[i].epp = fp->enclosures +i;

    rc = pthread_create(tid+i, NULL, run_query, qa+i);
    if (rc != 0) LF_ERROR(("creating thread for %s", fp->enclosures[i]->name));
  }

  for (i=0; i<fp->num_enclosures; ++i) {
    rc = pthread_join(tid[i], NULL);
    if (qa[i].rc != 0) {
       LF_ERROR(("query thread for %s failed", fp->enclosures[i]->name));
    }
  }

  return;

 except:
  exit(1);
}

static void *
run_query(
  void *v)
{
  struct qargs *ap = v;
  ap->rc = lf_query_switch(ap->name, ap->epp);
  pthread_exit(ap);
}
